#include <stdlib.h>
#include <string.h>
#include <poll.h>
#include <unistd.h>

#include "control/kmodules_loader.h"

#include "util/logger.h"
#include "util/kmodules_parser.h"
#include "control/sync_control.h"
#include "control/worker_thread_set.h"
#include "control/job_scheduler.h"

enum kml_state_t
{
	KML_STATE_DONE			= SYNC_CO_STATE_DONE,
	KML_STATE_WAITING_FOR_DEVICES	= SYNC_CO_STATE_WAITING_FOR_DEVICES,
	KML_STATE_LOADING_MODULES	= __SYNC_CO_LASTSTATE
};

static int state=(int)KML_STATE_DONE;

//main loop functions
static error_code_t kmodules_loader_mainloop(void);
static void kmodules_loader_prepare_poll(int *timeout_ptr, nfds_t *no_active_polls_ptr);
static int kmodules_loader_poll(struct pollfd *pollfds, nfds_t no_active_polls, int timeout);
static error_code_t kmodules_loader_dispatch_events(struct pollfd *pollfds, nfds_t no_active_polls);
static bool kmodules_loader_update_state(void);

//----------------------------------------------- modules loader api members ----------------------------------------------
error_code_t kmodules_loader_init(int argc,char *argv[])
{
	error_code_t result;
	logger_loglevel_t loglevel;

	loglevel=kmodules_parser_parse_loglevel(argc,argv);
	logger_init(loglevel);

	result=kmodule_init();

	if(result==RESULT_OK)
		result=job_scheduler_init();

	sync_control_init();

	if (result==RESULT_OK)
		result=kmodules_parser_parse_args(argc,argv);

	if (result==RESULT_OK)
		result=kmodules_parser_parse_config_early();

	if (result==RESULT_OK)
		result=worker_thread_set_init(kmodules_parser_get_max_worker_threads_cnt());

	logger_log_debug("KMODULES_LOADER -> Modules loader init done (result=%d).",result);
	return result;
}

error_code_t kmodules_loader_start(void)
{
	if (kmodules_parser_get_scheduled_modules_cnt()==0)
	{
		logger_log_error("No modules to be loaded. Please use device_sync instead!");
		return RESULT_INVALID_ARGS;
	}
	else
		return kmodules_loader_mainloop();
}

void kmodules_loader_deinit(bool dump_missing_files)
{  
	sync_control_deinit(dump_missing_files);
	worker_thread_set_deinit();
	job_scheduler_deinit();
	kmodule_deinit();

	logger_log_debug("KMODULES_LOADER -> loader deinit complete.");
}
//-------------------------------------------------------------------------------------------------------------------------

//------------------------------------------------modules loader private members ------------------------------------------

static void kmodules_loader_prepare_poll(int *timeout_ptr, nfds_t *no_active_polls_ptr)
{
	//disable device sync fd in case we have no devices to wait for any longer to prevent reacting
	//on any uevent coming in. The udev socket has been closed before as well so we don't have any fd to poll at all
	//in case we are not waiting for devices and we are not loading any modules, we are not using poll any longer.
	if ((state & KML_STATE_WAITING_FOR_DEVICES)==0)
	{
		if ((state & KML_STATE_LOADING_MODULES)==0)
			*no_active_polls_ptr=0;
		else
			*no_active_polls_ptr=1;
	}

	if ((state & KML_STATE_LOADING_MODULES)!=0)
	{
		//no timeouting while we are loading modules
		*timeout_ptr=-1;
	}
	else
	{
		//modules loaded, we now need a dynamic timeout
		*timeout_ptr=sync_control_get_remaining_timeout_time();
		//in case we are waiting for attributes only, we might need a smaller timeout
		if ((state & KML_STATE_WAITING_FOR_DEVICES)==0 && (*timeout_ptr > ATTRIBUTE_POLL_TIME_MS || *timeout_ptr==-1))
			*timeout_ptr=ATTRIBUTE_POLL_TIME_MS;
	}
}

static int kmodules_loader_poll(struct pollfd *pollfds, nfds_t no_active_polls, int timeout)
{
	if (no_active_polls==0)
	{
		usleep((unsigned int)(timeout*1000));
		return 0;
	}
	else
		return poll(pollfds,no_active_polls,timeout);
}

static error_code_t kmodules_loader_dispatch_events(struct pollfd *pollfds, nfds_t no_active_polls)
{
	error_code_t result=RESULT_OK;

	if (no_active_polls > 0)
	{
		if ((pollfds[0].revents & POLLIN) != 0)
		{
			//catch up results of worker threads
			result=worker_thread_set_on_event();
			if (result==RESULT_OK)
				//try to schedule new modules
				result=job_scheduler_process_modules();

			//enable timeouting when we are done with the modules
			if (result==RESULT_OK)
			{
				if (job_scheduler_is_done())
					sync_control_activate_timeout(kmodules_parser_get_timeout_ms());
			}
		}
	}

	if (no_active_polls > 1 && result==RESULT_OK)
	{
		if ((pollfds[1].revents & POLLIN) != 0)
			//react on uevents
			result=sync_control_on_event();
	}

	return result;
}

static bool kmodules_loader_update_state(void)
{
	state=sync_control_get_state();
	if (!job_scheduler_is_done())
		state |= KML_STATE_LOADING_MODULES;
	return state != KML_STATE_DONE;
}

static error_code_t kmodules_loader_mainloop(void)
{
	bool mainloop_active;
	struct pollfd pollfds[2];
	int poll_timeout;
	nfds_t no_active_polls;
	int poll_result;

	error_code_t error_status;


	logger_log_debug("KMODULES_LOADER -> Going to start program now with a maximum of %d threads.",
			kmodules_parser_get_max_worker_threads_cnt());

	//kick off modules loader to schedule first modules
	error_status=job_scheduler_process_modules();

	if (error_status==RESULT_OK)
		error_status=kmodules_parser_parse_config_late();

	//after having parsed the complete file we are sure that we are getting a valid event file descriptor
	//from sync_control in case we have to wait for devices. If no device are there to wait for, the main loop
	//will only use the first pollfd and ignore the second one.
	no_active_polls=2;
	pollfds[0].events=POLLIN;
	pollfds[0].fd=worker_thread_set_get_pollfd();
	pollfds[1].events=POLLIN;
	pollfds[1].fd=sync_control_get_pollfd();

	logger_log_debug("KMODULES_LOADER -> Entering the mainloop.");

	// now we know for sure if we have to wait for something. So we first update the state to inform the main loop
	// for what we actually have to wait
	mainloop_active=kmodules_loader_update_state();

	//if we don't have any modules to load anymore -> kick off timeouting
	if (job_scheduler_is_done())
		sync_control_activate_timeout(kmodules_parser_get_timeout_ms());


	while(mainloop_active && error_status==RESULT_OK)
	{
		kmodules_loader_prepare_poll(&poll_timeout,&no_active_polls);

		logger_log_debug("KMODULES_LOADER -> Polling on %d file descriptors, timeout: %d ms.",
				no_active_polls,poll_timeout);
		poll_result=kmodules_loader_poll(pollfds,no_active_polls,poll_timeout);
		logger_log_debug("KMODULES_LOADER -> Mainloop got woken up.");

		if (poll_result<0)
			continue;
		else if (poll_result==0)
		{
			if (sync_control_get_remaining_timeout_time()>0)
				error_status=sync_control_check_attributes();
			else
				error_status=RESULT_TIMEOUT_EXPIRED;
		}
		else
			error_status=kmodules_loader_dispatch_events(pollfds,no_active_polls);

		if (error_status == RESULT_OK)
			//checks the overall state and returns true when we are done completely
			mainloop_active=kmodules_loader_update_state();
	}

	logger_log_debug("KMODULES_LOADER -> Leaving the main loop. Result: %d",error_status);

	return error_status;
}
//-------------------------------------------------------------------------------------------------------------------------
